;-----------------------------------------------------------------------------;
; Fixed-point FFT routines for megaAVRs                        (C)ChaN, 2005
;-----------------------------------------------------------------------------;
;
; void fft_input (const int16_t *array_src, complex_t *array_bfly);
; void fft_execute (complex_t *array_bfly);
; void fft_output (complex_t *array_bfly, uint16_t *array_dst);
;
;  <array_src>: Wave form to be processed.
;  <array_bfly>: Complex array for butterfly operations.
;  <array_dst>: Spectrum output buffer.
;
; These functions must be called in sequence to do a DFT in FFT algorithm.
; fft_input() fills the complex array with a wave form to prepare butterfly
; operations. A hamming window is applied at the same time.
; fft_execute() executes the butterfly operations.
; fft_output() re-orders the results, converts the complex spectrum into
; scalar spectrum and output it in linear scale.
;
; The number of points FFT_N is defined in "ffft.h" and the value can be
; power of 2 in range of 64 - 1024.
;
;----------------------------------------------------------------------------;
; 16bit fixed-point FFT performance with MegaAVRs
; (Running at 16MHz/internal SRAM)
;
;  Points:   Input, Execute,  Output,    Total:  Throughput
;   64pts:   .17ms,   2.0ms,   1.2ms,    3.4ms:   19.0kpps
;  128pts:   .33ms,   4.6ms,   2.4ms,    7.3ms:   17.5kpps
;  256pts:   .66ms,  10.4ms,   4.9ms,   15.9ms:   16.1kpps
;  512pts:   1.3ms,  23.2ms,   9.7ms,   34.2ms:   14.9kpps
; 1024pts:   2.7ms,  51.7ms,  19.4ms,   73.7ms:   13.9kpps
;----------------------------------------------------------------------------;


.nolist
#define FFFT_ASM
#include "ffft.h"
.list

#define FFT_B_1024 10

;----------------------------------------------------------------------------;
; Constant Tables

.global tbl_window_1024
tbl_window_1024:	; tbl_window[] = ... (This is a Hamming window)
#if FFT_N_1024 == 1024
	.dc.w 2621, 2621, 2622, 2623, 2625, 2628, 2631, 2635, 2639, 2644, 2649, 2655, 2662, 2669, 2676, 2685
	.dc.w 2693, 2703, 2713, 2723, 2734, 2746, 2758, 2771, 2784, 2798, 2812, 2827, 2843, 2859, 2876, 2893
	.dc.w 2910, 2929, 2948, 2967, 2987, 3008, 3029, 3050, 3073, 3095, 3119, 3142, 3167, 3192, 3217, 3243
	.dc.w 3270, 3297, 3325, 3353, 3382, 3411, 3441, 3471, 3502, 3533, 3565, 3598, 3631, 3664, 3698, 3733
	.dc.w 3768, 3804, 3840, 3877, 3914, 3952, 3990, 4029, 4068, 4108, 4148, 4189, 4230, 4272, 4314, 4357
	.dc.w 4401, 4444, 4489, 4534, 4579, 4625, 4671, 4718, 4765, 4813, 4861, 4910, 4959, 5009, 5059, 5110
	.dc.w 5161, 5213, 5265, 5317, 5370, 5424, 5478, 5532, 5587, 5642, 5698, 5754, 5811, 5868, 5926, 5984
	.dc.w 6042, 6101, 6160, 6220, 6280, 6341, 6402, 6464, 6525, 6588, 6651, 6714, 6777, 6841, 6906, 6970
	.dc.w 7036, 7101, 7167, 7234, 7300, 7368, 7435, 7503, 7571, 7640, 7709, 7779, 7849, 7919, 7989, 8060
	.dc.w 8132, 8203, 8275, 8348, 8420, 8493, 8567, 8641, 8715, 8789, 8864, 8939, 9015, 9090, 9166, 9243
	.dc.w 9320, 9397, 9474, 9552, 9630, 9708, 9787, 9866, 9945, 10024, 10104, 10184, 10264, 10345, 10426, 10507
	.dc.w 10588, 10670, 10752, 10834, 10917, 10999, 11082, 11166, 11249, 11333, 11417, 11501, 11586, 11670, 11755, 11840
	.dc.w 11926, 12011, 12097, 12183, 12269, 12355, 12442, 12529, 12616, 12703, 12790, 12878, 12966, 13054, 13142, 13230
	.dc.w 13318, 13407, 13496, 13584, 13674, 13763, 13852, 13942, 14031, 14121, 14211, 14301, 14391, 14481, 14572, 14662
	.dc.w 14753, 14844, 14935, 15026, 15117, 15208, 15299, 15391, 15482, 15574, 15665, 15757, 15849, 15940, 16032, 16124
	.dc.w 16216, 16308, 16400, 16493, 16585, 16677, 16769, 16862, 16954, 17046, 17139, 17231, 17324, 17416, 17509, 17601
	.dc.w 17694, 17786, 17879, 17971, 18064, 18156, 18248, 18341, 18433, 18526, 18618, 18710, 18802, 18895, 18987, 19079
	.dc.w 19171, 19263, 19355, 19447, 19539, 19630, 19722, 19814, 19905, 19997, 20088, 20179, 20271, 20362, 20453, 20543
	.dc.w 20634, 20725, 20815, 20906, 20996, 21086, 21176, 21266, 21356, 21446, 21535, 21625, 21714, 21803, 21892, 21980
	.dc.w 22069, 22157, 22246, 22334, 22422, 22509, 22597, 22684, 22772, 22859, 22945, 23032, 23118, 23204, 23290, 23376
	.dc.w 23462, 23547, 23632, 23717, 23802, 23886, 23970, 24054, 24138, 24222, 24305, 24388, 24471, 24553, 24635, 24717
	.dc.w 24799, 24880, 24962, 25042, 25123, 25203, 25283, 25363, 25443, 25522, 25601, 25679, 25758, 25836, 25913, 25991
	.dc.w 26068, 26144, 26221, 26297, 26373, 26448, 26523, 26598, 26673, 26747, 26820, 26894, 26967, 27040, 27112, 27184
	.dc.w 27256, 27327, 27398, 27469, 27539, 27609, 27678, 27747, 27816, 27884, 27952, 28020, 28087, 28154, 28220, 28286
	.dc.w 28352, 28417, 28482, 28546, 28610, 28674, 28737, 28800, 28862, 28924, 28985, 29046, 29107, 29167, 29227, 29286
	.dc.w 29345, 29404, 29462, 29519, 29576, 29633, 29689, 29745, 29800, 29855, 29910, 29963, 30017, 30070, 30123, 30175
	.dc.w 30226, 30277, 30328, 30378, 30428, 30477, 30526, 30574, 30622, 30669, 30716, 30762, 30808, 30854, 30899, 30943
	.dc.w 30987, 31030, 31073, 31115, 31157, 31198, 31239, 31280, 31319, 31359, 31397, 31436, 31473, 31511, 31547, 31583
	.dc.w 31619, 31654, 31689, 31723, 31756, 31790, 31822, 31854, 31885, 31916, 31947, 31976, 32006, 32034, 32063, 32090
	.dc.w 32117, 32144, 32170, 32196, 32220, 32245, 32269, 32292, 32315, 32337, 32359, 32380, 32400, 32420, 32440, 32459
	.dc.w 32477, 32495, 32512, 32528, 32545, 32560, 32575, 32590, 32603, 32617, 32629, 32642, 32653, 32664, 32675, 32685
	.dc.w 32694, 32703, 32711, 32719, 32726, 32732, 32738, 32744, 32748, 32753, 32756, 32759, 32762, 32764, 32765, 32766
	.dc.w 32766, 32766, 32765, 32764, 32762, 32759, 32756, 32753, 32748, 32744, 32738, 32732, 32726, 32719, 32711, 32703
	.dc.w 32694, 32685, 32675, 32664, 32653, 32642, 32629, 32617, 32603, 32590, 32575, 32560, 32545, 32529, 32512, 32495
	.dc.w 32477, 32459, 32440, 32420, 32400, 32380, 32359, 32337, 32315, 32292, 32269, 32245, 32221, 32196, 32170, 32144
	.dc.w 32117, 32090, 32063, 32035, 32006, 31976, 31947, 31916, 31885, 31854, 31822, 31790, 31757, 31723, 31689, 31654
	.dc.w 31619, 31584, 31547, 31511, 31473, 31436, 31397, 31359, 31319, 31280, 31239, 31198, 31157, 31115, 31073, 31030
	.dc.w 30987, 30943, 30899, 30854, 30808, 30763, 30716, 30669, 30622, 30574, 30526, 30477, 30428, 30378, 30328, 30277
	.dc.w 30226, 30175, 30123, 30070, 30017, 29964, 29910, 29855, 29800, 29745, 29689, 29633, 29576, 29519, 29462, 29404
	.dc.w 29345, 29286, 29227, 29167, 29107, 29046, 28985, 28924, 28862, 28800, 28737, 28674, 28610, 28546, 28482, 28417
	.dc.w 28352, 28286, 28220, 28154, 28087, 28020, 27952, 27884, 27816, 27747, 27678, 27609, 27539, 27469, 27398, 27327
	.dc.w 27256, 27184, 27112, 27040, 26967, 26894, 26820, 26747, 26673, 26598, 26523, 26448, 26373, 26297, 26221, 26144
	.dc.w 26068, 25991, 25913, 25836, 25758, 25679, 25601, 25522, 25443, 25363, 25283, 25203, 25123, 25042, 24962, 24880
	.dc.w 24799, 24717, 24635, 24553, 24471, 24388, 24305, 24222, 24138, 24054, 23970, 23886, 23802, 23717, 23632, 23547
	.dc.w 23462, 23376, 23291, 23205, 23118, 23032, 22945, 22859, 22772, 22684, 22597, 22510, 22422, 22334, 22246, 22158
	.dc.w 22069, 21981, 21892, 21803, 21714, 21625, 21535, 21446, 21356, 21266, 21176, 21086, 20996, 20906, 20815, 20725
	.dc.w 20634, 20544, 20453, 20362, 20271, 20179, 20088, 19997, 19905, 19814, 19722, 19631, 19539, 19447, 19355, 19263
	.dc.w 19171, 19079, 18987, 18895, 18803, 18710, 18618, 18526, 18433, 18341, 18249, 18156, 18064, 17971, 17879, 17786
	.dc.w 17694, 17601, 17509, 17416, 17324, 17231, 17139, 17047, 16954, 16862, 16769, 16677, 16585, 16493, 16401, 16308
	.dc.w 16216, 16124, 16032, 15940, 15849, 15757, 15665, 15574, 15482, 15391, 15299, 15208, 15117, 15026, 14935, 14844
	.dc.w 14753, 14663, 14572, 14482, 14391, 14301, 14211, 14121, 14031, 13942, 13852, 13763, 13674, 13585, 13496, 13407
	.dc.w 13318, 13230, 13142, 13054, 12966, 12878, 12790, 12703, 12616, 12529, 12442, 12355, 12269, 12183, 12097, 12011
	.dc.w 11926, 11840, 11755, 11670, 11586, 11501, 11417, 11333, 11249, 11166, 11083, 11000, 10917, 10834, 10752, 10670
	.dc.w 10588, 10507, 10426, 10345, 10264, 10184, 10104, 10024, 9945, 9866, 9787, 9708, 9630, 9552, 9474, 9397
	.dc.w 9320, 9243, 9167, 9090, 9015, 8939, 8864, 8789, 8715, 8641, 8567, 8494, 8420, 8348, 8275, 8203
	.dc.w 8132, 8060, 7989, 7919, 7849, 7779, 7709, 7640, 7571, 7503, 7435, 7368, 7300, 7234, 7167, 7101
	.dc.w 7036, 6970, 6906, 6841, 6777, 6714, 6651, 6588, 6526, 6464, 6402, 6341, 6280, 6220, 6160, 6101
	.dc.w 6042, 5984, 5926, 5868, 5811, 5754, 5698, 5642, 5587, 5532, 5478, 5424, 5370, 5317, 5265, 5213
	.dc.w 5161, 5110, 5059, 5009, 4959, 4910, 4861, 4813, 4765, 4718, 4671, 4625, 4579, 4534, 4489, 4445
	.dc.w 4401, 4357, 4314, 4272, 4230, 4189, 4148, 4108, 4068, 4029, 3990, 3952, 3914, 3877, 3840, 3804
	.dc.w 3768, 3733, 3699, 3664, 3631, 3598, 3565, 3533, 3502, 3471, 3441, 3411, 3382, 3353, 3325, 3297
	.dc.w 3270, 3243, 3217, 3192, 3167, 3142, 3119, 3095, 3073, 3050, 3029, 3008, 2987, 2967, 2948, 2929
	.dc.w 2910, 2893, 2876, 2859, 2843, 2827, 2812, 2798, 2784, 2771, 2758, 2746, 2734, 2723, 2713, 2703
	.dc.w 2693, 2685, 2676, 2669, 2662, 2655, 2649, 2644, 2639, 2635, 2631, 2628, 2625, 2623, 2622, 2621
#endif


tbl_cos_sin:	; Table of {cos(x),sin(x)}, (0 <= x < pi, in FFT_N/2 steps)
#if FFT_N_1024 == 1024
	.dc.w	32767, 0, 32766, 201, 32764, 402, 32761, 603, 32757, 804, 32751, 1005, 32744, 1206, 32736, 1406
	.dc.w	32727, 1607, 32717, 1808, 32705, 2009, 32692, 2209, 32678, 2410, 32662, 2610, 32646, 2811, 32628, 3011
	.dc.w	32609, 3211, 32588, 3411, 32567, 3611, 32544, 3811, 32520, 4011, 32495, 4210, 32468, 4409, 32441, 4608
	.dc.w	32412, 4807, 32382, 5006, 32350, 5205, 32318, 5403, 32284, 5601, 32249, 5799, 32213, 5997, 32176, 6195
	.dc.w	32137, 6392, 32097, 6589, 32056, 6786, 32014, 6982, 31970, 7179, 31926, 7375, 31880, 7571, 31833, 7766
	.dc.w	31785, 7961, 31735, 8156, 31684, 8351, 31633, 8545, 31580, 8739, 31525, 8932, 31470, 9126, 31413, 9319
	.dc.w	31356, 9511, 31297, 9703, 31236, 9895, 31175, 10087, 31113, 10278, 31049, 10469, 30984, 10659, 30918, 10849
	.dc.w	30851, 11038, 30783, 11227, 30713, 11416, 30643, 11604, 30571, 11792, 30498, 11980, 30424, 12166, 30349, 12353
	.dc.w	30272, 12539, 30195, 12724, 30116, 12909, 30036, 13094, 29955, 13278, 29873, 13462, 29790, 13645, 29706, 13827
	.dc.w	29621, 14009, 29534, 14191, 29446, 14372, 29358, 14552, 29268, 14732, 29177, 14911, 29085, 15090, 28992, 15268
	.dc.w	28897, 15446, 28802, 15623, 28706, 15799, 28608, 15975, 28510, 16150, 28410, 16325, 28309, 16499, 28208, 16672
	.dc.w	28105, 16845, 28001, 17017, 27896, 17189, 27790, 17360, 27683, 17530, 27575, 17699, 27466, 17868, 27355, 18036
	.dc.w	27244, 18204, 27132, 18371, 27019, 18537, 26905, 18702, 26789, 18867, 26673, 19031, 26556, 19194, 26437, 19357
	.dc.w	26318, 19519, 26198, 19680, 26077, 19840, 25954, 20000, 25831, 20159, 25707, 20317, 25582, 20474, 25456, 20631
	.dc.w	25329, 20787, 25201, 20942, 25072, 21096, 24942, 21249, 24811, 21402, 24679, 21554, 24546, 21705, 24413, 21855
	.dc.w	24278, 22004, 24143, 22153, 24006, 22301, 23869, 22448, 23731, 22594, 23592, 22739, 23452, 22883, 23311, 23027
	.dc.w	23169, 23169, 23027, 23311, 22883, 23452, 22739, 23592, 22594, 23731, 22448, 23869, 22301, 24006, 22153, 24143
	.dc.w	22004, 24278, 21855, 24413, 21705, 24546, 21554, 24679, 21402, 24811, 21249, 24942, 21096, 25072, 20942, 25201
	.dc.w	20787, 25329, 20631, 25456, 20474, 25582, 20317, 25707, 20159, 25831, 20000, 25954, 19840, 26077, 19680, 26198
	.dc.w	19519, 26318, 19357, 26437, 19194, 26556, 19031, 26673, 18867, 26789, 18702, 26905, 18537, 27019, 18371, 27132
	.dc.w	18204, 27244, 18036, 27355, 17868, 27466, 17699, 27575, 17530, 27683, 17360, 27790, 17189, 27896, 17017, 28001
	.dc.w	16845, 28105, 16672, 28208, 16499, 28309, 16325, 28410, 16150, 28510, 15975, 28608, 15799, 28706, 15623, 28802
	.dc.w	15446, 28897, 15268, 28992, 15090, 29085, 14911, 29177, 14732, 29268, 14552, 29358, 14372, 29446, 14191, 29534
	.dc.w	14009, 29621, 13827, 29706, 13645, 29790, 13462, 29873, 13278, 29955, 13094, 30036, 12909, 30116, 12724, 30195
	.dc.w	12539, 30272, 12353, 30349, 12166, 30424, 11980, 30498, 11792, 30571, 11604, 30643, 11416, 30713, 11227, 30783
	.dc.w	11038, 30851, 10849, 30918, 10659, 30984, 10469, 31049, 10278, 31113, 10087, 31175, 9895, 31236, 9704, 31297
	.dc.w	9511, 31356, 9319, 31413, 9126, 31470, 8933, 31525, 8739, 31580, 8545, 31633, 8351, 31684, 8156, 31735
	.dc.w	7961, 31785, 7766, 31833, 7571, 31880, 7375, 31926, 7179, 31970, 6983, 32014, 6786, 32056, 6589, 32097
	.dc.w	6392, 32137, 6195, 32176, 5997, 32213, 5799, 32249, 5601, 32284, 5403, 32318, 5205, 32350, 5006, 32382
	.dc.w	4807, 32412, 4608, 32441, 4409, 32468, 4210, 32495, 4011, 32520, 3811, 32544, 3611, 32567, 3411, 32588
	.dc.w	3211, 32609, 3011, 32628, 2811, 32646, 2610, 32662, 2410, 32678, 2209, 32692, 2009, 32705, 1808, 32717
	.dc.w	1607, 32727, 1407, 32736, 1206, 32744, 1005, 32751, 804, 32757, 603, 32761, 402, 32764, 201, 32766
	.dc.w	0, 32766, -201, 32766, -402, 32764, -603, 32761, -804, 32757, -1005, 32751, -1206, 32744, -1406, 32736
	.dc.w	-1607, 32727, -1808, 32717, -2009, 32705, -2209, 32692, -2410, 32678, -2610, 32662, -2811, 32646, -3011, 32628
	.dc.w	-3211, 32609, -3411, 32588, -3611, 32567, -3811, 32544, -4010, 32520, -4210, 32495, -4409, 32468, -4608, 32441
	.dc.w	-4807, 32412, -5006, 32382, -5205, 32350, -5403, 32318, -5601, 32284, -5799, 32249, -5997, 32213, -6195, 32176
	.dc.w	-6392, 32137, -6589, 32097, -6786, 32056, -6982, 32014, -7179, 31970, -7375, 31926, -7571, 31880, -7766, 31833
	.dc.w	-7961, 31785, -8156, 31735, -8351, 31684, -8545, 31633, -8739, 31580, -8932, 31525, -9126, 31470, -9319, 31413
	.dc.w	-9511, 31356, -9703, 31297, -9895, 31236, -10087, 31175, -10278, 31113, -10469, 31049, -10659, 30984, -10849, 30918
	.dc.w	-11038, 30851, -11227, 30783, -11416, 30713, -11604, 30643, -11792, 30571, -11979, 30498, -12166, 30424, -12353, 30349
	.dc.w	-12539, 30272, -12724, 30195, -12909, 30116, -13094, 30036, -13278, 29955, -13462, 29873, -13645, 29790, -13827, 29706
	.dc.w	-14009, 29621, -14191, 29534, -14372, 29446, -14552, 29358, -14732, 29268, -14911, 29177, -15090, 29085, -15268, 28992
	.dc.w	-15446, 28897, -15623, 28802, -15799, 28706, -15975, 28608, -16150, 28510, -16325, 28410, -16499, 28309, -16672, 28208
	.dc.w	-16845, 28105, -17017, 28001, -17189, 27896, -17360, 27790, -17530, 27683, -17699, 27575, -17868, 27466, -18036, 27355
	.dc.w	-18204, 27244, -18371, 27132, -18537, 27019, -18702, 26905, -18867, 26789, -19031, 26673, -19194, 26556, -19357, 26438
	.dc.w	-19519, 26318, -19680, 26198, -19840, 26077, -20000, 25954, -20159, 25831, -20317, 25707, -20474, 25582, -20631, 25456
	.dc.w	-20787, 25329, -20942, 25201, -21096, 25072, -21249, 24942, -21402, 24811, -21554, 24679, -21705, 24546, -21855, 24413
	.dc.w	-22004, 24278, -22153, 24143, -22301, 24006, -22448, 23869, -22594, 23731, -22739, 23592, -22883, 23452, -23027, 23311
	.dc.w	-23169, 23169, -23311, 23027, -23452, 22883, -23592, 22739, -23731, 22594, -23869, 22448, -24006, 22301, -24143, 22153
	.dc.w	-24278, 22005, -24413, 21855, -24546, 21705, -24679, 21554, -24811, 21402, -24942, 21249, -25072, 21096, -25201, 20942
	.dc.w	-25329, 20787, -25456, 20631, -25582, 20474, -25707, 20317, -25831, 20159, -25954, 20000, -26077, 19840, -26198, 19680
	.dc.w	-26318, 19519, -26437, 19357, -26556, 19194, -26673, 19031, -26789, 18867, -26905, 18702, -27019, 18537, -27132, 18371
	.dc.w	-27244, 18204, -27355, 18036, -27466, 17868, -27575, 17699, -27683, 17530, -27790, 17360, -27896, 17189, -28001, 17017
	.dc.w	-28105, 16845, -28207, 16672, -28309, 16499, -28410, 16325, -28510, 16150, -28608, 15975, -28706, 15799, -28802, 15623
	.dc.w	-28897, 15446, -28992, 15268, -29085, 15090, -29177, 14911, -29268, 14732, -29358, 14552, -29446, 14372, -29534, 14191
	.dc.w	-29620, 14009, -29706, 13827, -29790, 13645, -29873, 13462, -29955, 13278, -30036, 13094, -30116, 12910, -30195, 12724
	.dc.w	-30272, 12539, -30349, 12353, -30424, 12167, -30498, 11980, -30571, 11792, -30643, 11604, -30713, 11416, -30783, 11228
	.dc.w	-30851, 11038, -30918, 10849, -30984, 10659, -31049, 10469, -31113, 10278, -31175, 10087, -31236, 9895, -31297, 9704
	.dc.w	-31356, 9511, -31413, 9319, -31470, 9126, -31525, 8933, -31580, 8739, -31633, 8545, -31684, 8351, -31735, 8156
	.dc.w	-31784, 7961, -31833, 7766, -31880, 7571, -31926, 7375, -31970, 7179, -32014, 6983, -32056, 6786, -32097, 6589
	.dc.w	-32137, 6392, -32175, 6195, -32213, 5997, -32249, 5799, -32284, 5601, -32318, 5403, -32350, 5205, -32382, 5006
	.dc.w	-32412, 4807, -32441, 4609, -32468, 4409, -32495, 4210, -32520, 4011, -32544, 3811, -32567, 3611, -32588, 3411
	.dc.w	-32609, 3211, -32628, 3011, -32646, 2811, -32662, 2611, -32678, 2410, -32692, 2210, -32705, 2009, -32717, 1808
	.dc.w	-32727, 1607, -32736, 1407, -32744, 1206, -32751, 1005, -32757, 804, -32761, 603, -32764, 402, -32766, 201
#endif



tbl_bitrev:		; tbl_bitrev[] = ...
#if FFT_N_1024 == 1024
#ifdef INPUT_IQ
	.dc.w	1*4, 513*4, 257*4, 769*4, 129*4, 641*4, 385*4, 897*4, 65*4, 577*4, 321*4, 833*4, 193*4, 705*4, 449*4, 961*4
	.dc.w	33*4, 545*4, 289*4, 801*4, 161*4, 673*4, 417*4, 929*4, 97*4, 609*4, 353*4, 865*4, 225*4, 737*4, 481*4, 993*4
	.dc.w	17*4, 529*4, 273*4, 785*4, 145*4, 657*4, 401*4, 913*4, 81*4, 593*4, 337*4, 849*4, 209*4, 721*4, 465*4, 977*4
	.dc.w	49*4, 561*4, 305*4, 817*4, 177*4, 689*4, 433*4, 945*4, 113*4, 625*4, 369*4, 881*4, 241*4, 753*4, 497*4, 1009*4
	.dc.w	9*4, 521*4, 265*4, 777*4, 137*4, 649*4, 393*4, 905*4, 73*4, 585*4, 329*4, 841*4, 201*4, 713*4, 457*4, 969*4
	.dc.w	41*4, 553*4, 297*4, 809*4, 169*4, 681*4, 425*4, 937*4, 105*4, 617*4, 361*4, 873*4, 233*4, 745*4, 489*4, 1001*4
	.dc.w	25*4, 537*4, 281*4, 793*4, 153*4, 665*4, 409*4, 921*4, 89*4, 601*4, 345*4, 857*4, 217*4, 729*4, 473*4, 985*4
	.dc.w	57*4, 569*4, 313*4, 825*4, 185*4, 697*4, 441*4, 953*4, 121*4, 633*4, 377*4, 889*4, 249*4, 761*4, 505*4, 1017*4
	.dc.w	5*4, 517*4, 261*4, 773*4, 133*4, 645*4, 389*4, 901*4, 69*4, 581*4, 325*4, 837*4, 197*4, 709*4, 453*4, 965*4
	.dc.w	37*4, 549*4, 293*4, 805*4, 165*4, 677*4, 421*4, 933*4, 101*4, 613*4, 357*4, 869*4, 229*4, 741*4, 485*4, 997*4
	.dc.w	21*4, 533*4, 277*4, 789*4, 149*4, 661*4, 405*4, 917*4, 85*4, 597*4, 341*4, 853*4, 213*4, 725*4, 469*4, 981*4
	.dc.w	53*4, 565*4, 309*4, 821*4, 181*4, 693*4, 437*4, 949*4, 117*4, 629*4, 373*4, 885*4, 245*4, 757*4, 501*4, 1013*4
	.dc.w	13*4, 525*4, 269*4, 781*4, 141*4, 653*4, 397*4, 909*4, 77*4, 589*4, 333*4, 845*4, 205*4, 717*4, 461*4, 973*4
	.dc.w	45*4, 557*4, 301*4, 813*4, 173*4, 685*4, 429*4, 941*4, 109*4, 621*4, 365*4, 877*4, 237*4, 749*4, 493*4, 1005*4
	.dc.w	29*4, 541*4, 285*4, 797*4, 157*4, 669*4, 413*4, 925*4, 93*4, 605*4, 349*4, 861*4, 221*4, 733*4, 477*4, 989*4
	.dc.w	61*4, 573*4, 317*4, 829*4, 189*4, 701*4, 445*4, 957*4, 125*4, 637*4, 381*4, 893*4, 253*4, 765*4, 509*4, 1021*4
	.dc.w	3*4, 515*4, 259*4, 771*4, 131*4, 643*4, 387*4, 899*4, 67*4, 579*4, 323*4, 835*4, 195*4, 707*4, 451*4, 963*4
	.dc.w	35*4, 547*4, 291*4, 803*4, 163*4, 675*4, 419*4, 931*4, 99*4, 611*4, 355*4, 867*4, 227*4, 739*4, 483*4, 995*4
	.dc.w	19*4, 531*4, 275*4, 787*4, 147*4, 659*4, 403*4, 915*4, 83*4, 595*4, 339*4, 851*4, 211*4, 723*4, 467*4, 979*4
	.dc.w	51*4, 563*4, 307*4, 819*4, 179*4, 691*4, 435*4, 947*4, 115*4, 627*4, 371*4, 883*4, 243*4, 755*4, 499*4, 1011*4
	.dc.w	11*4, 523*4, 267*4, 779*4, 139*4, 651*4, 395*4, 907*4, 75*4, 587*4, 331*4, 843*4, 203*4, 715*4, 459*4, 971*4
	.dc.w	43*4, 555*4, 299*4, 811*4, 171*4, 683*4, 427*4, 939*4, 107*4, 619*4, 363*4, 875*4, 235*4, 747*4, 491*4, 1003*4
	.dc.w	27*4, 539*4, 283*4, 795*4, 155*4, 667*4, 411*4, 923*4, 91*4, 603*4, 347*4, 859*4, 219*4, 731*4, 475*4, 987*4
	.dc.w	59*4, 571*4, 315*4, 827*4, 187*4, 699*4, 443*4, 955*4, 123*4, 635*4, 379*4, 891*4, 251*4, 763*4, 507*4, 1019*4
	.dc.w	7*4, 519*4, 263*4, 775*4, 135*4, 647*4, 391*4, 903*4, 71*4, 583*4, 327*4, 839*4, 199*4, 711*4, 455*4, 967*4
	.dc.w	39*4, 551*4, 295*4, 807*4, 167*4, 679*4, 423*4, 935*4, 103*4, 615*4, 359*4, 871*4, 231*4, 743*4, 487*4, 999*4
	.dc.w	23*4, 535*4, 279*4, 791*4, 151*4, 663*4, 407*4, 919*4, 87*4, 599*4, 343*4, 855*4, 215*4, 727*4, 471*4, 983*4
	.dc.w	55*4, 567*4, 311*4, 823*4, 183*4, 695*4, 439*4, 951*4, 119*4, 631*4, 375*4, 887*4, 247*4, 759*4, 503*4, 1015*4
	.dc.w	15*4, 527*4, 271*4, 783*4, 143*4, 655*4, 399*4, 911*4, 79*4, 591*4, 335*4, 847*4, 207*4, 719*4, 463*4, 975*4
	.dc.w	47*4, 559*4, 303*4, 815*4, 175*4, 687*4, 431*4, 943*4, 111*4, 623*4, 367*4, 879*4, 239*4, 751*4, 495*4, 1007*4
	.dc.w	31*4, 543*4, 287*4, 799*4, 159*4, 671*4, 415*4, 927*4, 95*4, 607*4, 351*4, 863*4, 223*4, 735*4, 479*4, 991*4
	.dc.w	63*4, 575*4, 319*4, 831*4, 191*4, 703*4, 447*4, 959*4, 127*4, 639*4, 383*4, 895*4, 255*4, 767*4, 511*4, 1023*4
#endif
	.dc.w	0*4, 512*4, 256*4, 768*4, 128*4, 640*4, 384*4, 896*4, 64*4, 576*4, 320*4, 832*4, 192*4, 704*4, 448*4, 960*4
	.dc.w	32*4, 544*4, 288*4, 800*4, 160*4, 672*4, 416*4, 928*4, 96*4, 608*4, 352*4, 864*4, 224*4, 736*4, 480*4, 992*4
	.dc.w	16*4, 528*4, 272*4, 784*4, 144*4, 656*4, 400*4, 912*4, 80*4, 592*4, 336*4, 848*4, 208*4, 720*4, 464*4, 976*4
	.dc.w	48*4, 560*4, 304*4, 816*4, 176*4, 688*4, 432*4, 944*4, 112*4, 624*4, 368*4, 880*4, 240*4, 752*4, 496*4, 1008*4
	.dc.w	8*4, 520*4, 264*4, 776*4, 136*4, 648*4, 392*4, 904*4, 72*4, 584*4, 328*4, 840*4, 200*4, 712*4, 456*4, 968*4
	.dc.w	40*4, 552*4, 296*4, 808*4, 168*4, 680*4, 424*4, 936*4, 104*4, 616*4, 360*4, 872*4, 232*4, 744*4, 488*4, 1000*4
	.dc.w	24*4, 536*4, 280*4, 792*4, 152*4, 664*4, 408*4, 920*4, 88*4, 600*4, 344*4, 856*4, 216*4, 728*4, 472*4, 984*4
	.dc.w	56*4, 568*4, 312*4, 824*4, 184*4, 696*4, 440*4, 952*4, 120*4, 632*4, 376*4, 888*4, 248*4, 760*4, 504*4, 1016*4
	.dc.w	4*4, 516*4, 260*4, 772*4, 132*4, 644*4, 388*4, 900*4, 68*4, 580*4, 324*4, 836*4, 196*4, 708*4, 452*4, 964*4
	.dc.w	36*4, 548*4, 292*4, 804*4, 164*4, 676*4, 420*4, 932*4, 100*4, 612*4, 356*4, 868*4, 228*4, 740*4, 484*4, 996*4
	.dc.w	20*4, 532*4, 276*4, 788*4, 148*4, 660*4, 404*4, 916*4, 84*4, 596*4, 340*4, 852*4, 212*4, 724*4, 468*4, 980*4
	.dc.w	52*4, 564*4, 308*4, 820*4, 180*4, 692*4, 436*4, 948*4, 116*4, 628*4, 372*4, 884*4, 244*4, 756*4, 500*4, 1012*4
	.dc.w	12*4, 524*4, 268*4, 780*4, 140*4, 652*4, 396*4, 908*4, 76*4, 588*4, 332*4, 844*4, 204*4, 716*4, 460*4, 972*4
	.dc.w	44*4, 556*4, 300*4, 812*4, 172*4, 684*4, 428*4, 940*4, 108*4, 620*4, 364*4, 876*4, 236*4, 748*4, 492*4, 1004*4
	.dc.w	28*4, 540*4, 284*4, 796*4, 156*4, 668*4, 412*4, 924*4, 92*4, 604*4, 348*4, 860*4, 220*4, 732*4, 476*4, 988*4
	.dc.w	60*4, 572*4, 316*4, 828*4, 188*4, 700*4, 444*4, 956*4, 124*4, 636*4, 380*4, 892*4, 252*4, 764*4, 508*4, 1020*4
	.dc.w	2*4, 514*4, 258*4, 770*4, 130*4, 642*4, 386*4, 898*4, 66*4, 578*4, 322*4, 834*4, 194*4, 706*4, 450*4, 962*4
	.dc.w	34*4, 546*4, 290*4, 802*4, 162*4, 674*4, 418*4, 930*4, 98*4, 610*4, 354*4, 866*4, 226*4, 738*4, 482*4, 994*4
	.dc.w	18*4, 530*4, 274*4, 786*4, 146*4, 658*4, 402*4, 914*4, 82*4, 594*4, 338*4, 850*4, 210*4, 722*4, 466*4, 978*4
	.dc.w	50*4, 562*4, 306*4, 818*4, 178*4, 690*4, 434*4, 946*4, 114*4, 626*4, 370*4, 882*4, 242*4, 754*4, 498*4, 1010*4
	.dc.w	10*4, 522*4, 266*4, 778*4, 138*4, 650*4, 394*4, 906*4, 74*4, 586*4, 330*4, 842*4, 202*4, 714*4, 458*4, 970*4
	.dc.w	42*4, 554*4, 298*4, 810*4, 170*4, 682*4, 426*4, 938*4, 106*4, 618*4, 362*4, 874*4, 234*4, 746*4, 490*4, 1002*4
	.dc.w	26*4, 538*4, 282*4, 794*4, 154*4, 666*4, 410*4, 922*4, 90*4, 602*4, 346*4, 858*4, 218*4, 730*4, 474*4, 986*4
	.dc.w	58*4, 570*4, 314*4, 826*4, 186*4, 698*4, 442*4, 954*4, 122*4, 634*4, 378*4, 890*4, 250*4, 762*4, 506*4, 1018*4
	.dc.w	6*4, 518*4, 262*4, 774*4, 134*4, 646*4, 390*4, 902*4, 70*4, 582*4, 326*4, 838*4, 198*4, 710*4, 454*4, 966*4
	.dc.w	38*4, 550*4, 294*4, 806*4, 166*4, 678*4, 422*4, 934*4, 102*4, 614*4, 358*4, 870*4, 230*4, 742*4, 486*4, 998*4
	.dc.w	22*4, 534*4, 278*4, 790*4, 150*4, 662*4, 406*4, 918*4, 86*4, 598*4, 342*4, 854*4, 214*4, 726*4, 470*4, 982*4
	.dc.w	54*4, 566*4, 310*4, 822*4, 182*4, 694*4, 438*4, 950*4, 118*4, 630*4, 374*4, 886*4, 246*4, 758*4, 502*4, 1014*4
	.dc.w	14*4, 526*4, 270*4, 782*4, 142*4, 654*4, 398*4, 910*4, 78*4, 590*4, 334*4, 846*4, 206*4, 718*4, 462*4, 974*4
	.dc.w	46*4, 558*4, 302*4, 814*4, 174*4, 686*4, 430*4, 942*4, 110*4, 622*4, 366*4, 878*4, 238*4, 750*4, 494*4, 1006*4
	.dc.w	30*4, 542*4, 286*4, 798*4, 158*4, 670*4, 414*4, 926*4, 94*4, 606*4, 350*4, 862*4, 222*4, 734*4, 478*4, 990*4
	.dc.w	62*4, 574*4, 318*4, 830*4, 190*4, 702*4, 446*4, 958*4, 126*4, 638*4, 382*4, 894*4, 254*4, 766*4, 510*4, 1022*4
#endif



;----------------------------------------------------------------------------;
#ifndef INPUT_NOISE
.global fft_input_1024
.func fft_input_1024
fft_input_1024:
	pushw	T2H,T2L
	pushw	AH,AL
	pushw	YH,YL

	movw	XL, EL				;X = array_src;
	movw	YL, DL				;Y = array_bfly;
	clr	EH				;Zero
	ldiw	ZH,ZL, tbl_window_1024		;Z = &tbl_window[0];
	ldiw	AH,AL, FFT_N_1024			;A = FFT_N;
1:	lpmw	BH,BL, Z+			;B = *Z++; (window)
	ldw	CH,CL, X+			;C = *X++; (I-axis)
	FMULS16	DH,DL,T2H,T2L, BH,BL, CH,CL	;D = B * C;
	stw	Y+, DH,DL			;*Y++ = D;
#ifdef INPUT_IQ
	ldw	CH,CL, X+			;C = *X++; (Q-axis)
	FMULS16	DH,DL,T2H,T2L, BH,BL, CH,CL	;D = B * C;
#endif
	stw	Y+, DH,DL			;*Y++ = D;
	subiw	AH,AL, 1			;while(--A)
	brne	1b				;/

	popw	YH,YL
	popw	AH,AL
	popw	T2H,T2L
	clr	r1
	ret
.endfunc
#endif	/* INPUT_NOISE */



;----------------------------------------------------------------------------;
.global fft_execute_1024
.func fft_execute_1024
fft_execute_1024:
	pushw	T2H,T2L
	pushw	T4H,T4L
	pushw	T6H,T6L
	pushw	T8H,T8L
	pushw	T10H,T10L
	pushw	T12H,T12L
	pushw	T14H,T14L
	pushw	AH,AL
	pushw	YH,YL

	movw	ZL, EL				;Z = array_bfly;
	ldiw	EH,EL, 1			;E = 1;
	ldiw	XH,XL, FFT_N_1024/2			;X = FFT_N/2;
1:	ldi	AL, 4				;T12 = E; (angular speed)
	mul	EL, AL				;
	movw	T12L, T0L			;
	mul	EH, AL				;
	add	T12H, T0L			;/
	movw	T14L, EL			;T14 = E;
	pushw	EH,EL
	movw	YL, ZL				;Z = &array_bfly[0];
	mul	XL, AL				;Y = &array_bfly[X];
	addw	YH,YL, T0H,T0L			;
	mul	XH, AL				;
	add	YH, T0L				;/
	pushw	ZH,ZL
2:	clrw	T10H,T10L			;T10 = 0 (angle)
	clr	EH				;Zero reg.
3:	lddw	AH,AL, Z+0			;A = *Z - *Y; *Z++ += *Y;
	asrw	AH,AL				;
	lddw	DH,DL, Y+0			;
	asrw	DH,DL				;
	movw	CL, AL				;
	subw	AH,AL, DH,DL			;
	addw	CH,CL, DH,DL			;
	stw	Z+, CH,CL			;/
	lddw	BH,BL, Z+0			;B = *Z - *Y; *Z++ += *Y;
	asrw	BH,BL				;
	lddw	DH,DL, Y+2			;
	asrw	DH,DL				;
	movw	CL, BL				;
	subw	BH,BL, DH,DL			;
	addw	CH,CL, DH,DL			;
	stw	Z+, CH,CL			;/
	movw	T0L, ZL
	ldiw	ZH,ZL, tbl_cos_sin		;C = cos(T10); D = sin(T10);
	addw	ZH,ZL, T10H,T10L		;
	lpmw	CH,CL, Z+			;
	lpmw	DH,DL, Z+			;/
	movw	ZL, T0L
	FMULS16	T4H,T4L,T2H,T2L, AH,AL, CH,CL	;*Y++ = A * C + B * D;
	FMULS16	T8H,T8L,T6H,T6L, BH,BL, DH,DL	;
	addd	T4H,T4L,T2H,T2L, T8H,T8L,T6H,T6L;
	stw	Y+, T4H,T4L			;/
	FMULS16	T4H,T4L,T2H,T2L, BH,BL, CH,CL 	;*Y++ = B * C - A * D;
	FMULS16	T8H,T8L,T6H,T6L, AH,AL, DH,DL 	;
	subd	T4H,T4L,T2H,T2L, T8H,T8L,T6H,T6L;
	stw	Y+, T4H,T4L			;/
	addw	T10H,T10L, T12H,T12L		;T10 += T12; (next angle)
#if FFT_N_1024 >= 128
	sbrs	T10H, FFT_B_1024 - 7			;while(T10 < pi)
#else
	sbrs	T10L, FFT_B_1024 + 1
#endif
	rjmp	3b				;/
	ldi	AL, 4				;Y += X; Z += X; (skip split segment)
	mul	XL, AL
	addw	YH,YL, T0H,T0L			;
	addw	ZH,ZL, T0H,T0L			;
	mul	XH, AL				;
	add	YH, T0L				;
	add	ZH, T0L				;/
	ldi	EL, 1				;while(--T14)
	subw	T14H,T14L, EH,EL		;
	rjne	2b				;/
	popw	ZH,ZL
	popw	EH,EL
	lslw	EH,EL				;E *= 2;
	lsrw	XH,XL				;while(X /= 2)
	adiw	XL, 0				;
	rjne	1b				;/

	popw	YH,YL
	popw	AH,AL
	popw	T14H,T14L
	popw	T12H,T12L
	popw	T10H,T10L
	popw	T8H,T8L
	popw	T6H,T6L
	popw	T4H,T4L
	popw	T2H,T2L
;	clr	r1
	ret
.endfunc



;----------------------------------------------------------------------------;
.global fft_output_1024
.func fft_output_1024
fft_output_1024:
	pushw	T2H,T2L
	pushw	T4H,T4L
	pushw	T6H,T6L
	pushw	T8H,T8L
	pushw	T10H,T10L
	pushw	AH,AL
	pushw	YH,YL

	movw	T10L, EL			;T10 = array_bfly;
	movw	YL, DL				;Y = array_output;
	ldiw	ZH,ZL, tbl_bitrev		;Z = tbl_bitrev;
	clr	EH				;Zero
#ifdef INPUT_IQ
	ldiw	AH,AL, FFT_N_1024			;A = FFT_N; (plus/minus)
#else
	ldiw	AH,AL, FFT_N_1024 / 2		;A = FFT_N / 2; (plus only)
#endif
1:	lpmw	XH,XL, Z+			;X = *Z++;
	addw	XH,XL, T10H,T10L		;X += array_bfly;
	ldw	BH,BL, X+			;B = *X++;
	ldw	CH,CL, X+			;C = *X++;
	FMULS16	T4H,T4L,T2H,T2L, BH,BL, BH,BL	;T4:T2 = B * B;
	FMULS16	T8H,T8L,T6H,T6L, CH,CL, CH,CL	;T8:T6 = C * C;
	addd	T4H,T4L,T2H,T2L, T8H,T8L,T6H,T6L;T4:T2 += T8:T6;
	SQRT32					;B = sqrt(T4:T2);
	stw	Y+, BH,BL			;*Y++ = B;
	subiw	AH,AL, 1			;while(--A)
	rjne	1b				;/

	popw	YH,YL
	popw	AH,AL
	popw	T10H,T10L
	popw	T8H,T8L
	popw	T6H,T6L
	popw	T4H,T4L
	popw	T2H,T2L
	clr	r1
	ret
.endfunc



;----------------------------------------------------------------------------;
.global fmuls_f_1024
.func fmuls_f_1024
fmuls_f_1024:
	movw	CL, EL				;C = E;
	clr	EH	;Zero
	FMULS16	ZH,ZL,XH,XL, CH,CL, DH,DL	;Z:X = C * D;
	movw	EL, ZL
	clr	r1
	ret
.endfunc

